

  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Wed Nov 17 17:25:55 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief A brief description of this module
   *
   *  \version CRYS_DES.c#1:csrc:6
   *  \author adams
   *  \remarks Copyright (C) 2004 by Discretix Technologies Ltd.
   *           All Rights reserved
   */



/************* Include Files ****************/

/* .............. CRYS level includes ................. */

#include "DX_VOS_Mem.h"
#include "CRYS.h"
#include "CRYS_DES_error.h"
#include "dx_hw_defs.h"
#include "crys_host_op_code.h"

/************************ Defines ******************************/


/* canceling the PC-lint warning:
   Use of goto is deprecated */
/*lint --e{801} */

/************************ Enums ******************************/


/************************ MACROS ******************************/

/* this macro is required to remove compilers warnings if the DES is not supported */

#ifdef CRYS_NO_DES_SUPPORT
#define RETURN_IF_DES_UNSUPPORTED( a , b , c , d , e , f , g , h , i , j ) \
  (a)=(a);(b)=(b);(c)=(c);(d)=(d);(e)=(e);(f)=(f);(g)=(g);(h)=(h);(i)=(i);(j)=(j); \
  return CRYS_DES_IS_NOT_SUPPORTED
#else  /* !CRYS_NO_DES_SUPPORT */
#define RETURN_IF_DES_UNSUPPORTED( a , b , c , d , e , f , g , h , i , j ) 
#endif /* !CRYS_NO_DES_SUPPORT */

/************************ Global Data ******************************/

/************* Private function prototype ****************/


/************************ Public Functions ******************************/

/* ------------------------------------------------------------
 * @brief This function is used to initialize the DES machine.
 *        In order to operate the DES machine the first function that should be
 *        called is this function.
 *
 *        The function executes the following major steps:
 *
 *        1. Validates all of the inputs of the function. If one of the received 
 *           parameters is not valid it shall return an error:
 *
 *           - verifying that the context pointer is not DX_NULL (*ContextID_ptr).
 *           - verifying the pointer of the IV counter is not DX_NULL 
 *             if the modes that is selected are CBC.
 *           - verifying that the pointer to the key buffer is not DX_NULL 
 *           - verifying the values of the number of keys is valid ( 0- 2 ).
 *           - verifying the value of the operation mode is valid ( 0 - 2 ) 
 *        2. Decrypting the received context to the working context after capturing 
 *           the working context by calling the CRYS_CCM_GetContext() call.
 *
 *        3. Initializing the working context by the following:
 *           - loading the keys.
 *           - loading the IV counters.
 *           - loading the control field 
 *           - loading the operation mode. 
 *           - loading the DecryptEncrypt flag and the key size.
 *           - loading the key size.
 *        4. Encrypting the information in the working context and storing 
 *           it to the users received context. After then the working context is released.
 *           This state is operated by calling the CRYS_CCM_EncryptAndReleaseContext call.
 *        5. Exit the handler with the OK code.
 *
 *
 * @param[in] ContextID_ptr - a pointer to the DES context buffer allocated by the user that
 *                       is used for the DES machine operation.
 *
 * @param[in,out] IV_ptr - this parameter is the buffer of the IV or counters on mode CTR.
 *                          On ECB mode this parameter has no use.
 *                          On CBC this parameter should containe the IV values.
 *
 * @param[in] Key_ptr -  a pointer to the users key buffer.
 *
 * @param[in] NumOfKeys - the number of keys used by the module ( 1 - 3 )
 *
 * @param[in] EncryptDecryptFlag - This flag determains if the DES shall perform an Encrypt operation [0] or a
 *                           Decrypt operation [1].
 *
 * @param[in] OperationMode - The operation mode : ECB or CBC.
 * 
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 */

CEXPORT_C CRYSError_t  CRYS_DES_Init( 
                            CRYS_DESUserContext_t   *ContextID_ptr,
                            CRYS_DES_Iv_t            IV_ptr,
                            CRYS_DES_Key_t          *Key_ptr,
                            CRYS_DES_NumOfKeys_t     NumOfKeys,
                            CRYS_DES_EncryptMode_t   EncryptDecryptFlag,
                            CRYS_DES_OperationMode_t OperationMode )
{
	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_DES_INIT_OP_CODE, 6,
	   							 /* PARAMETERS: */
								 ContextID_ptr,
								 IV_ptr,
								 Key_ptr,
								 NumOfKeys,
								 EncryptDecryptFlag,
								 OperationMode );
   
   
}/* END OF DES_Init */   
                         
/** ------------------------------------------------------------
 * @brief This function is used to operate a block on the DES machine.
 *        This function should be called after the CRYS_DES_Init function
 *        was called.
 *
 *        The function executes the following major steps:
 *
 *        1.Checks the validation of all of the inputs of the function.
 *          If one of the received parameters is not valid it shall return an error.
 *
 *        2.Decrypting the received context to the working context after 
 *          capturing the working context by calling the CRYS_CCM_GetContext() call.
 *
 *        3.executing the DES operation on the hardware by calling the 
 *          low level DES function LLF_DES_Block.
 *        4.Encrypting the information in the working context and storing it 
 *          to the users received context. After then the working context is released.
 *          This state is operated by calling the CRYS_CCM_ReleaseContext call.
 *        5.Exit the handler with the OK code.  
 *      
 *
 * @param[in] ContextID_ptr - a pointer to the DES context buffer allocated by the user that
 *                       is used for the DES machine operation. this should be the same context that was
 *                       used on the previous call of this session.
 *
 * @param[in] DataIn_ptr - The pointer to the buffer of the input data to the DES. The pointer does 
 *                         not need to be aligned. On CSI input mode the pointer must be equal to
 *                         value (0xFFFFFFFC | DataInAlignment). 
 *
 * @param[in] DataInSize - The size of the input data in bytes: must be not 0 and must be multiple 
 *                         of 8 bytes.
 *
 * @param[in/out] DataOut_ptr - The pointer to the buffer of the output data from the DES. The pointer does not 
 *                              need to be aligned. On CSI output mode the pointer must be equal to
 *                              value (0xFFFFFFFC | DataOutAlignment).  
 *
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 *
 *     NOTE: Temporarily it is not allowed, that both the Input and the Output simultaneously 
 *           were on CSI mode
 */
 CEXPORT_C CRYSError_t  CRYS_DES_Block( 
                              CRYS_DESUserContext_t       *ContextID_ptr,   
                              DxUint8_t                     *DataIn_ptr,     
                              DxUint32_t                    DataInSize,     
                              DxUint8_t                     *DataOut_ptr )
 {                              
	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_DES_BLOCK_OP_CODE, 4,
	   							 /* PARAMETERS: */
								 ContextID_ptr,
								 DataIn_ptr, DataInSize,
								 DataOut_ptr );
   
}/* END OF CRYS_DES_Block */ 

/**
 * @brief This function is used to end the DES operation seesion.
 *        It is the last function called on the DES operation.
 *
 *
 *        The function executes the following major steps:
 *
 *        1. Checks the validity of all of the inputs of the function. 
 *           If one of the received parameters is not valid it shall return an error.
 *
 *           The major checkers that are run over the received parameters:
 *           - verifying that the context pointer is not DX_NULL (*ContextID_ptr). 
 *        2. Clearing the users context.
 *        3. Exit the handler with the OK code.
 *      
 *
 * @param[in] ContextID_ptr - a pointer to the DES context buffer allocated by the user that
 *                       is used for the DES machine operation. this should be the same context that was
 *                       used on the previous call of this session.
 *
 *
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 */
CEXPORT_C CRYSError_t  CRYS_DES_Free(CRYS_DESUserContext_t  *ContextID_ptr )
{
  /* The return error identifier */
  CRYSError_t Error;

  /*-----------------------
      CODE
  --------------------------*/
  Error = CRYS_OK;
                          
  /* ............... checking the parameters validity ................... */
  
  /* if the users context ID pointer is DX_NULL return an error */
  if( ContextID_ptr == DX_NULL )
  {
    Error = CRYS_DES_INVALID_USER_CONTEXT_POINTER_ERROR;
    goto end_function;
  } 
    
  /* .............. clearing the users context .......................... */ 
  DX_VOS_MemSet( ContextID_ptr , 0 , sizeof(CRYS_DESUserContext_t) );   

  /* ................. end of function ..................................... */
  /* ----------------------------------------------------------------------- */
    
end_function:
   
  return Error; 
   
}/* END OF CRYS_DES_Free */

/**
 * @brief This function is used to operate the DES machine in one integrated operation.
 *
 *        The actual macros that will be used by the users are:
 *      
 *
 * @param[in,out] IVCounter_ptr - this parameter is the buffer of the IV or counters on mode CTR.
 *                          On ECB mode this parameter has no use.
 *                          On CBC mode this parameter should containe the IV values.
 *
 * @param[in] Key_ptr - a pointer to the users key buffer.
 *
 * @param[in] KeySize - Thenumber of keys used by the DES as defined in the enum.
 *
 * @param[in] EncryptDecryptFlag - This flag determains if the DES shall perform an Encrypt operation [0] or a
 *                           Decrypt operation [1].
 *
 * @param[in] OperationMode - The operation mode : ECB or CBC.
 *
 * @param[in] DataIn_ptr - The pointer to the buffer of the input data to the DES. The pointer does 
 *                         not need to be aligned. On CSI input mode the pointer must be equal to
 *                         value (0xFFFFFFFC | DataInAlignment). 
 *
 * @param[in] DataInSize - The size of the input data in bytes: must be not 0 and must be multiple 
 *                         of 8 bytes.
 *
 * @param[in/out] DataOut_ptr - The pointer to the buffer of the output data from the DES. The pointer does not 
 *                              need to be aligned. On CSI output mode the pointer must be equal to
 *                              value (0xFFFFFFFC | DataOutAlignment).  
 *
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 *
 *     NOTES: 1. Temporarily it is not allowed, that both the Input and the Output simultaneously 
 *               were on CSI mode.
 *            2. Temporarily the CSI input or output are not allowed on XCBC, CMAC and XTS modes.    
 *
 */
 CEXPORT_C CRYSError_t  CRYS_DES(
                CRYS_DES_Iv_t             IV_ptr,                 /* in */ 
                CRYS_DES_Key_t           *Key_ptr,                /* in */ 
                CRYS_DES_NumOfKeys_t      NumOfKeys,              /* in */ 
                CRYS_DES_EncryptMode_t    EncryptDecryptFlag,     /* in */  
                CRYS_DES_OperationMode_t  OperationMode,          /* in */     
                DxUint8_t                *DataIn_ptr,             /* in */ 
                DxUint32_t                DataInSize,             /* in */ 
                DxUint8_t                *DataOut_ptr )           /* out */ 
{    

	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_DES_OP_CODE, 8,
	   							 /* PARAMETERS: */
								 IV_ptr,
								 Key_ptr,
								 NumOfKeys,
								 EncryptDecryptFlag,
								 OperationMode,
								 DataIn_ptr, DataInSize,
								 DataOut_ptr );
   
}/* END OF CRYS_DES */   

